using System.Data;
using System.Net;
using System.Net.Sockets;
using System.Reflection;
using System.Windows.Forms;
using gov.va.med.vbecs.Common;
using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.VistaLinkConfig;
using VistALinkControlTable = gov.va.med.vbecs.Common.VbecsTables.VistaLinkControl;
using VbecsConfigTable = gov.va.med.vbecs.Common.VbecsTables.VbecsGlobalConfig; // CR 3419
using Sprocs = gov.va.med.vbecs.Common.VbecsStoredProcs; // CR 3419
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/17/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Container for VBECS' VistALink configuration parameters. Provides validation, retrieve/update 
	///	capabilities and conversion to VistALink native connection info type.
	///</summary>

	#endregion

	public class VistALinkConfig : BaseBusinessObject
	{
		private const int MinTcpPortNumber = 1024;
		private const int MaxTcpPortNumber = 65535;

		private IPAddress _vistaIpAddress;
		private string _vistaDomainName;
		private int _vistaTcpPort;

        //CR 3419 start
        private IPAddress _vbecsServiceIpAddress;
        private int _vbecsServicePortNumber;
        private bool _vistaLinkConnectionIsDirty; // This solves the issue with the save button only enabling when VistALink is tested
        private bool _vistaLinkServiceIsDirty; // We must know when the service has changed so we can restart it
        private bool _vistaLinkTestedSuccessfully;
        // CR 3419 end

		private const string PrimaryServerHandle = "Primary";
		private const string PrimaryServerAlias = "Primary VBECS-connected VistA System";

		/// <summary>
		/// Constructs fresh instance of the class.
		/// </summary>
		private VistALinkConfig()
		{			
			SetInitialBrokenRules( BrRules.FormClassName, BrRules.RuleSets.BaseRuleSet );

			this.VistaPortNumber = StNullConvert.Int32Null;
			this.VistaIpAddress = null;
			this.VistaDomainName = null;
            this.VbecsServiceIpAddress = null;                      // CR 3419
            this.VbecsServicePortNumber = StNullConvert.Int32Null;  // CR 3419

			this.IsNew = true;
			this.IsDirty = false;

            this.VistaLinkConnectionIsDirty = false;    // CR 3419
            this.VistaLinkServiceIsDirty = false;       // CR 3419

		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5585"> 
		///		<ExpectedInput>None, VistALink configuration is defined in the DB.</ExpectedInput>
		///		<ExpectedOutput>Object initialized with non-null values from DB and not marked as new, no BRs are broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5591"> 
		///		<ExpectedInput>None, VistALink configuration is not defined in the DB (fresh install).</ExpectedInput>
		///		<ExpectedOutput>Object is empty, marked as new, 2 BRs are broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5586"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets current VistALink configuration by retrieving it from DB or 
		/// constructing a fresh object if there's no configuration available in the DB.
		/// </summary>
		/// <returns>An instance of VistALink config representing current VistALink configuration.</returns>
		public static VistALinkConfig GetCurrent()
		{
			VistALinkConfig _resultConfig = new VistALinkConfig();

			DataRow _currentValues = DAL.VistALinkConfig.Load( PrimaryServerHandle );

			if( _currentValues != null )
				_resultConfig.LoadFromDataRow( _currentValues );

			return _resultConfig;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5583"> 
		///		<ExpectedInput>Valid DataRow, object is valid.</ExpectedInput>
		///		<ExpectedOutput>Data is loaded into the supplied DataRow.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5584"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5592"> 
		///		<ExpectedInput>Valid DataRow, but object is invalid.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			if( !IsValid ) 
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToSaveInvalidBusinessObject( typeof( VistALinkConfig ).Name ).ResString ) );

			Utility.RequireColumns( dtRow, VistALinkControlTable.ServerAppHandle, VistALinkControlTable.ServerAlias, VistALinkControlTable.VistaPortNumber, VistALinkControlTable.RowVersion );

			dtRow[ VistALinkControlTable.ServerAppHandle ] = StDbNullConvert.From( PrimaryServerHandle );
			dtRow[ VistALinkControlTable.ServerAlias ] = StDbNullConvert.From( PrimaryServerAlias );			
			dtRow[ VistALinkControlTable.VistaIpAddress ] = StDbNullConvert.From( this.VistaIpAddress );
			dtRow[ VistALinkControlTable.VistaDomainName ] = StDbNullConvert.From( this.VistaDomainName );
			dtRow[ VistALinkControlTable.VistaPortNumber ] = StDbNullConvert.From( this.VistaPortNumber );
            dtRow[VistALinkControlTable.VbecsServiceIpAddress] = StDbNullConvert.From(this.VbecsServiceIpAddress); // CR 3419
            dtRow[VistALinkControlTable.VbecsServicePortNumber] = StDbNullConvert.From(this.VbecsServicePortNumber); // CR 3419
			dtRow[ VistALinkControlTable.RowVersion ] = StDbNullConvert.From( this.RowVersion );

			return dtRow;
		}

		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			Utility.RequireNonNullColumns( dtRow, VistALinkControlTable.VistaPortNumber, VistALinkControlTable.RowVersion );

			this.VistaIpAddress = StDbNullConvert.ToIPAddress( dtRow[ VistALinkControlTable.VistaIpAddress ] );
			this.VistaDomainName = StDbNullConvert.ToString( dtRow[ VistALinkControlTable.VistaDomainName ] );
			this.VistaPortNumber = StDbNullConvert.ToInt32( dtRow[ VistALinkControlTable.VistaPortNumber ] );
            this.VbecsServiceIpAddress = StDbNullConvert.ToIPAddress(dtRow[VistALinkControlTable.VbecsServiceIpAddress]); // CR 3419
            this.VbecsServicePortNumber = StDbNullConvert.ToInt32(dtRow[VistALinkControlTable.VbecsServicePortNumber]); // CR 3419
			this.RowVersion = StDbNullConvert.ToRowversion( dtRow[ VistALinkControlTable.RowVersion ] );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.InvalidDataLoadedFromDataRowIntoBusinessObject( typeof( VistALinkConfig ).Name, MethodBase.GetCurrentMethod().Name ).ResString ) );

			this.IsNew = false;
			this.IsDirty = false;

            this.VistaLinkConnectionIsDirty = false; // CR 3419
            this.VistaLinkServiceIsDirty = false; // CR 3419
		}
		
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/17/2004</CreationDate>
		///<TestCases>	
		///	
		///<Case type="0" testid ="5581"> 
		///		<ExpectedInput>Update function, object is valid and config already exists in the DB.</ExpectedInput>
		///		<ExpectedOutput>Object data is saved to the DB, object is not new and dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5593"> 
		///		<ExpectedInput>Update function, object is valid and config is not in the DB yet.</ExpectedInput>
		///		<ExpectedOutput>Object data is saved to the DB, object is not new and dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5582"> 
		///		<ExpectedInput>Update function, object is not valid (BRs are broken).</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves server connection info to the database.
		/// </summary>
		///	<param name="updateFunction">Update function.</param>
		public virtual void Save( UpdateFunction updateFunction )
		{
			DataRow _bufferRow = DAL.VistALinkConfig.GetVistaLinkControlEmptyDataTable().NewRow();
			LoadDataRowFromThis( _bufferRow );

			_bufferRow[ VistALinkControlTable.LastUpdateFunctionId ] = (int)updateFunction;

			DAL.VistALinkConfig.Save( _bufferRow );

			IsNew = false;
			IsDirty = false;

            this.VistaLinkConnectionIsDirty = false; // CR 3419
		}

		/// <summary>
		/// Triggered whenever any significant object property is changed.
		/// </summary>
		protected virtual void OnPropertyChanged()
		{
			this.IsDirty = true;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5579"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Server connection info object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5580"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Converts the connection parameters to VistALink native <see cref="ServerConnectionInfo"/> data type.
		/// </summary>
		/// <returns>New instance of <see cref="ServerConnectionInfo"/> containing VistALink connection parameters.</returns>
		public virtual ServerConnectionInfo GetServerConnectionInfo()
		{
            if (!IsVistaServerConnectionValid()) 
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToExportVistALinkConfigurationFromBO( this.GetType().Name ).ResString ) );

            //CR3414: if domain name is provided, it is translated to IP address. 
            var ipAddress = _vistaIpAddress ?? GetIpbyHostName(_vistaDomainName, _vistaTcpPort);
            //In case of domain name we already called GetIpbyHostName which checks connection to a domain name
            //In case of IP address we have to do additional check for connectivity
		    if (_vistaIpAddress != null)
		    {
               TestConnecteivity(new IPEndPoint(_vistaIpAddress, _vistaTcpPort));
		    }
		    return new ServerConnectionInfo(PrimaryServerHandle, PrimaryServerAlias, ipAddress, _vistaTcpPort);
		}

        // CR3419: We added new fields to the form for the VBECS VistALinkService. This method helps determine if 
        // there are rules broken related to the original VistA connection fields, not the new one.
        private bool IsVistaServerConnectionValid()
        {
            if (IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerIpAddressAndDomainNameBothSpecified) ||
                IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerIpAddressAndDomainNameNotSpecified) ||
                IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerIpAddressNotSet) ||
                IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerPortNumberNotSet) ||
                IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerPortNumberOutOfRange))
                return false;
            else
                return true;
        }

        //makes sure that new connection may be opened to end point
	    private static void TestConnecteivity(IPEndPoint endPoint)
	    {
             using (var tcpClient = new TcpClient())
                {
                    tcpClient.Connect(endPoint);
                    if (!tcpClient.Connected)
                    {
                        throw (new BusinessObjectException(StrRes.SysErrMsg.Common.VistALinkNetworkFailure().ResString));
                    }
                }
	    }

	    // Get IP address by domain
	    private static IPAddress GetIpbyHostName(string hostname, int port)
	    {
	        if (string.IsNullOrEmpty(hostname))
	        {
	            throw (new BusinessObjectException(
                    StrRes.SysErrMsg.Common.RequiredFieldMissing(MethodBase.GetCurrentMethod().Name).ResString));
	        }
            using (var tcpClient = new TcpClient(hostname, port))
	        {
	            if (!tcpClient.Connected)
	            {
	                throw (new BusinessObjectException(StrRes.SysErrMsg.Common.VistALinkNetworkFailure().ResString));
	            }

	            return ((IPEndPoint) tcpClient.Client.RemoteEndPoint).Address;
	        }
	    }

	    ///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5577"> 
		///		<ExpectedInput>Valid IP address.</ExpectedInput>
		///		<ExpectedOutput>No rules are broken, value is assigned, object marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5594"> 
		///		<ExpectedInput>Null value.</ExpectedInput>
		///		<ExpectedOutput>Corresponding rule is broken, object marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA system IP address.
		/// </summary>
		public virtual IPAddress VistaIpAddress
		{
			get
			{
				return _vistaIpAddress;
			}
			set
			{
				if (_vistaIpAddress == value || ( _vistaIpAddress != null && value != null & _vistaIpAddress.Equals( value ))) return;

				_vistaIpAddress = value;
                this.VistaLinkConnectionIsDirty = true; // CR 3419
                this.VistaLinkTestedSuccessfully = false; // CR 3419

				// MBR_1.12 
				RuleBroken(BrRules.VistaServerIpAddressAndDomainNameNotSpecified, StNullConvert.IsNull( _vistaIpAddress )&& (_vistaDomainName == null || _vistaDomainName.Trim() == string.Empty));

				if( _vistaIpAddress == value || ( _vistaIpAddress != null && value != null & _vistaIpAddress.Equals( value ) ) )

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>B. Tomin, C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/9/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8492"> 
		///		<ExpectedInput>Valid domain name value</ExpectedInput>
		///		<ExpectedOutput>No rules are broken, object marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8493"> 
		///		<ExpectedInput>Undefined domain name</ExpectedInput>
		///		<ExpectedOutput>Corresponding rule is broken, object marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA M server domain name
		/// </summary>
		public virtual string VistaDomainName
		{
			get
			{
				return _vistaDomainName;
			}
			set
			{
				if (_vistaDomainName == value) return;

				_vistaDomainName = value;

                this.VistaLinkConnectionIsDirty = true; // CR 3419
                this.VistaLinkTestedSuccessfully = false; // CR 3419


				// MBR_1.12 
				RuleBroken(BrRules.VistaServerIpAddressAndDomainNameNotSpecified, StNullConvert.IsNull( _vistaIpAddress )&& (_vistaDomainName == null || _vistaDomainName.Trim() == string.Empty));

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5575"> 
		///		<ExpectedInput>Valid port number value.</ExpectedInput>
		///		<ExpectedOutput>No rules are broken, object marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5595"> 
		///		<ExpectedInput>Undefined port number.</ExpectedInput>
		///		<ExpectedOutput>Corresponding rule is broken, object marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5576"> 
		///		<ExpectedInput>Out-of-range port number.</ExpectedInput>
		///		<ExpectedOutput>Corresponding rule is broken, object marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VistA system TCP port number. 
		/// </summary>
		public virtual int VistaPortNumber
		{
			get
			{
				return _vistaTcpPort;
			}
			set
			{
				if( _vistaTcpPort == value )
					return;

                this.VistaLinkConnectionIsDirty = true; // CR 3419
                this.VistaLinkTestedSuccessfully = false; // CR 3419


				_vistaTcpPort = value;

				bool _tcpPortIsDefined = !RuleBroken( BrRules.VistaServerPortNumberNotSet, StNullConvert.IsNull( _vistaTcpPort ) );
			
				RuleBroken( BrRules.VistaServerPortNumberOutOfRange, 
					_tcpPortIsDefined && ( _vistaTcpPort < MinTcpPortNumber || _vistaTcpPort > MaxTcpPortNumber ) );

				OnPropertyChanged();
			}
		}


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2014</CreationDate>
        // CR 3419
        public virtual IPAddress VbecsServiceIpAddress
        {
            get
            {
                return _vbecsServiceIpAddress;
            }
            set
            {
                if (_vbecsServiceIpAddress == value || (_vbecsServiceIpAddress != null && value != null & _vbecsServiceIpAddress.Equals(value))) return;

                _vbecsServiceIpAddress = value;
                VistaLinkServiceIsDirty = true; // CR 3419

                // MBR_1. (CR 3419 TBD)
                RuleBroken(BrRules.VbecsServiceIpAddressNotSet, StNullConvert.IsNull(_vbecsServiceIpAddress));

                if (_vbecsServiceIpAddress == value || (_vbecsServiceIpAddress != null && value != null & _vbecsServiceIpAddress.Equals(value)))

                    OnPropertyChanged();
            }
        }


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2014</CreationDate>
        ///<summary>Port Number for the VBECS RPC XML Listenr Service - CR3419</summary>
        public virtual int VbecsServicePortNumber
        {
            get
            {
                return _vbecsServicePortNumber;
            }
            set
            {
                if (_vbecsServicePortNumber == value)
                    return;

                _vbecsServicePortNumber = value;
                VistaLinkServiceIsDirty = true; // CR 3419

                bool _tcpPortIsDefined = !RuleBroken(BrRules.VbecsServicePortNumberNotSet, StNullConvert.IsNull(_vbecsServicePortNumber));

                RuleBroken(BrRules.VbecsServicePortNumberOutOfRange,
                    _tcpPortIsDefined && (_vbecsServicePortNumber < MinTcpPortNumber || _vbecsServicePortNumber > MaxTcpPortNumber));

                OnPropertyChanged();
            }
        }



        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2014</CreationDate>
        ///<summary>Has the vistalink connection changed since last tested - CR3419</summary>
        public bool VistaLinkConnectionIsDirty
        {
            set
            {
                _vistaLinkConnectionIsDirty = value;
            }
            get
            {
                return _vistaLinkConnectionIsDirty;
            }
        }

        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2014</CreationDate>
        ///CR 3419
        public bool VistaLinkServiceIsDirty
        {
            set
            {
                _vistaLinkServiceIsDirty = value;
            }
            get
            {
                return _vistaLinkServiceIsDirty;
            }
        }

        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2014</CreationDate>
        ///<summary>VistaLink Connection Tested - CR3419</summary>
        public bool VistaLinkTestedSuccessfully
        {
            set
            {
                _vistaLinkTestedSuccessfully = value;
            }
            get
            {
                return _vistaLinkTestedSuccessfully;
            }
        }


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2014</CreationDate>
        ///<summary>CR3419</summary>
        public bool IsVistALinkConnectionConfigValid
        {
            get
            {
                if
                (
                    this.IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerIpAddressAndDomainNameBothSpecified) ||
                    this.IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerIpAddressAndDomainNameNotSpecified) ||
                    this.IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerIpAddressNotSet) ||
                    this.IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerPortNumberNotSet) ||
                    this.IsRuleBroken(Common.VbecsBrokenRules.VistaLinkConfig.VistaServerPortNumberOutOfRange)
                )
                {
                    return false;
                }
                else
                {
                    return true;
                }
            }
        }
	}
}
